<?php

namespace App\Http\Controllers;

use App\Models\ServiceOrder;
use Illuminate\Http\Request;


class ServiceOrderController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $allowedRoles = ['compras', 'receptoria', 'agente_aduanal', 'administrador'];

        if (!in_array($user->role, $allowedRoles)) {
            abort(403, 'Acceso denegado');
        }

        $orders = ServiceOrder::latest()->get();
        return view('service-orders.index', compact('orders'));
    }

    public function create()
    {
        if (auth()->user()->role !== 'receptoria') {
            abort(403, 'Acceso denegado');
        }
        return view('service-orders.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'cliente' => 'required',
            'descripcion' => 'required',
            'estado' => 'required|in:pendiente,completado',
        ]);

        ServiceOrder::create($data);

        return redirect()->route('pedidos-de-servicio.index')
            ->with('success', '¡Orden creada con éxito!');
    }

    public function edit(ServiceOrder $serviceOrder)
    {
        if (auth()->user()->role !== 'receptoria') {
            abort(403, 'Acceso denegado');
        }
        return view('service-orders.edit', compact('serviceOrder'));
    }

    public function update(Request $request, ServiceOrder $serviceOrder)
    {
        $data = $request->validate([
            'cliente' => 'required',
            'descripcion' => 'required',
            'estado' => 'required|in:pendiente,completado',
        ]);

        $serviceOrder->update($data);

        return redirect()->route('pedidos-de-servicio.index')
            ->with('success', '¡Orden actualizada con éxito!');
    }

    public function destroy(ServiceOrder $serviceOrder)
    {
        $user = auth()->user();

        if ($user->role !== 'compras' || !$user->approved) {
            abort(403, 'Acceso denegado');
        }

        $serviceOrder->delete();

        if (request()->expectsJson()) {
            return response()->json(['message' => '¡Orden eliminada con éxito!']);
        }

        return redirect()->route('pedidos-de-servicio.index')
            ->with('success', '¡Orden eliminada con éxito!');
    }
}
